//
//  ViewController.swift
//  SwiftAVFCIDetector
//
//  Created by Yoshihisa Nitta on 2016/06/21.
//  Copyright © 2016年 Yoshihisa Nitta. All rights reserved.
//

import UIKit
import AVFoundation

class ViewController: UIViewController, AVCaptureVideoDataOutputSampleBufferDelegate {
    
    var detector: CIDetector!
    var maskImage: UIImage!
    var startDate: NSDate!
    
    var mySession: AVCaptureSession!
    var myCamera: AVCaptureDevice!
    var myVideoInput: AVCaptureDeviceInput!
    var myVideoOutput: AVCaptureVideoDataOutput!
    var detectFlag: Bool = false

    @IBOutlet weak var myImageView: UIImageView!
    @IBAction func tapStart(sender: AnyObject) {
        mySession.startRunning()
    }
    @IBAction func tapStop(sender: AnyObject) {
        mySession.stopRunning()
    }
    @IBAction func tapDetect(sender: AnyObject) {
        detectFlag = !detectFlag
    }
    func captureOutput(captureOutput: AVCaptureOutput!, didOutputSampleBuffer sampleBuffer: CMSampleBuffer!, fromConnection connection: AVCaptureConnection!) {
        print("captureOutput:didOutputSampleBuffer:fromConnection)")
        if connection.supportsVideoOrientation {
            connection.videoOrientation = AVCaptureVideoOrientation.Portrait
        }
        dispatch_async(dispatch_get_main_queue(), {
            //dispatch_sync(dispatch_get_main_queue(), {
            let image = self.imageFromSampleBuffer(sampleBuffer)
            if self.detectFlag {
                self.myImageView.image = self.detectFace(image)
            } else {
                self.myImageView.image = image
            }
        })
    }
    
    func imageFromSampleBuffer(sampleBuffer: CMSampleBufferRef) -> UIImage {
        let imageBuffer: CVImageBufferRef = CMSampleBufferGetImageBuffer(sampleBuffer)!
        CVPixelBufferLockBaseAddress(imageBuffer, 0)   // Lock Base Address
        let baseAddress = CVPixelBufferGetBaseAddressOfPlane(imageBuffer, 0)  // Get Original Image Information
        let bytesPerRow = CVPixelBufferGetBytesPerRow(imageBuffer)
        let width = CVPixelBufferGetWidth(imageBuffer)
        let height = CVPixelBufferGetHeight(imageBuffer)
        
        let colorSpace = CGColorSpaceCreateDeviceRGB()  // RGB ColorSpace
        let bitmapInfo = (CGBitmapInfo.ByteOrder32Little.rawValue | CGImageAlphaInfo.PremultipliedFirst.rawValue)
        let context = CGBitmapContextCreate(baseAddress, width, height, 8, bytesPerRow, colorSpace, bitmapInfo)
        let imageRef = CGBitmapContextCreateImage(context) // Create Quarts image
        
        CVPixelBufferUnlockBaseAddress(imageBuffer, 0)    // Unlock Base Address
        
        let resultImage: UIImage = UIImage(CGImage: imageRef!)
        
        return resultImage
    }
    
    func prepareVideo() {
        mySession = AVCaptureSession()
        mySession.sessionPreset = AVCaptureSessionPresetHigh
        let devices = AVCaptureDevice.devices()
        for device in devices {
            if (device.position == AVCaptureDevicePosition.Back) {
                myCamera = device as! AVCaptureDevice
            }
        }
        do {
            myVideoInput = try AVCaptureDeviceInput(device: myCamera)
            if (mySession.canAddInput(myVideoInput)) {
                mySession.addInput(myVideoInput)
            } else {
                print("cannot add input to session")
            }
            
            myVideoOutput = AVCaptureVideoDataOutput()
            myVideoOutput.videoSettings = [kCVPixelBufferPixelFormatTypeKey : Int(kCVPixelFormatType_32BGRA)]
            myVideoOutput.setSampleBufferDelegate(self,queue:dispatch_get_main_queue())
            //myVideoOutput.setSampleBufferDelegate(self,queue:dispatch_queue_create("myqueue",nil))
            myVideoOutput.alwaysDiscardsLateVideoFrames = true
            if (mySession.canAddOutput(myVideoOutput)) {
                mySession.addOutput(myVideoOutput)
            } else {
                print("cannot add output to session")
            }
            
            /* // preview background
             let myVideoLayer = AVCaptureVideoPreviewLayer(session: mySession)
             myVideoLayer.frame = view.bounds
             myVideoLayer.videoGravity = AVLayerVideoGravityResizeAspectFill
             view.layer.insertSublayer(myVideoLayer,atIndex:0)
             */
        } catch let error as NSError {
            print("cannot use camera \(error)")
        }
    }
    
    func detectFace(image: UIImage) -> UIImage {
        let deltaTime:Double = NSDate().timeIntervalSinceDate(startDate)
        let modTime = deltaTime - Double(Int(deltaTime/4) * 4)
        let mask = rotateImage(maskImage!, Float(3.1415 * modTime))
        
        let ciImage:CIImage! = CIImage(image: image)
        let options = [CIDetectorSmile:true]
        let features = detector.featuresInImage(ciImage, options:options)
        UIGraphicsBeginImageContext(image.size);
        image.drawInRect(CGRectMake(0,0,image.size.width,image.size.height))
        let context: CGContextRef = UIGraphicsGetCurrentContext()!
        CGContextSetLineWidth(context, 5.0);
        CGContextSetRGBStrokeColor(context, 0.0, 1.0, 0.0, 1.0)
        for feature in features as! [CIFaceFeature] {
            var rect:CGRect = feature.bounds;
            rect.origin.y = image.size.height - rect.origin.y - rect.height;
            if feature.hasSmile {
                CGContextAddRect(context, rect);
                CGContextStrokePath(context)
            } else {
                CGContextDrawImage(context,rect,mask.CGImage)
            }
            if feature.hasTrackingID {
                let str = NSAttributedString(string: String(feature.trackingID),
                                             attributes:[NSForegroundColorAttributeName: UIColor.redColor(),
                                                NSFontAttributeName: UIFont.boldSystemFontOfSize(16.0)])
                str.drawAtPoint(CGPointMake(rect.origin.x, rect.origin.y-30))
            }
        }
        let img = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        
        return img;
    }
    
    func rotateImage(image: UIImage, _ radian: Float) -> UIImage {
        let size: CGSize = image.size
        UIGraphicsBeginImageContext(size);
        let context = UIGraphicsGetCurrentContext()
        CGContextTranslateCTM(context,size.width/2, size.height/2) // rotation center
        CGContextScaleCTM(context, 1.0, -1.0) // flip y-coordinate
        CGContextRotateCTM(context, CGFloat(-radian))
        CGContextDrawImage(UIGraphicsGetCurrentContext(),CGRectMake(-size.width/2,-size.height/2, size.width, size.height), image.CGImage)
        let img = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return img
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        let options:[String:AnyObject] = [CIDetectorAccuracy:CIDetectorAccuracyHigh]
        //let options:[String:AnyObject] = [CIDetectorAccuracy:CIDetectorAccuracyHigh, CIDetectorTracking:true]
        detector = CIDetector(ofType: CIDetectorTypeFace, context: nil, options: options)
        maskImage = UIImage(named: "LaughingMan.png")
        startDate = NSDate()
        prepareVideo()
    }

    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }

}

